
'use client';

import React, { useState } from 'react';
import { motion } from 'framer-motion';
import { Mail, MapPin, Send, User, MessageSquare, Phone, Building2, Globe, Layers } from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';

function ContactForm() {
  const { content } = useLanguage();
  const { contact: contactContent } = content;
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    message: ''
  });
  const [focused, setFocused] = useState('');

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    console.log('Form submitted:', formData);
    // Here you would typically send the form data to a server
  };

  const inputClasses = "w-full px-4 py-3 rounded-xl bg-white/50 border-2 border-slate-200 focus:border-blue-500 focus:bg-white transition-all duration-300 outline-none";

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <motion.div
        initial={{ opacity: 0, x: -20 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ delay: 0.1 }}
      >
        <div className="relative">
          <User className={`absolute left-3 top-3.5 w-5 h-5 transition-colors duration-300 ${focused === 'name' ? 'text-blue-600' : 'text-slate-400'}`} />
          <input
            type="text"
            placeholder={contactContent.fields.name}
            className={`${inputClasses} pl-11`}
            value={formData.name}
            onChange={(e) => setFormData({...formData, name: e.target.value})}
            onFocus={() => setFocused('name')}
            onBlur={() => setFocused('')}
          />
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, x: -20 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ delay: 0.2 }}
      >
        <div className="relative">
          <Mail className={`absolute left-3 top-3.5 w-5 h-5 transition-colors duration-300 ${focused === 'email' ? 'text-purple-600' : 'text-slate-400'}`} />
          <input
            type="email"
            placeholder={contactContent.fields.email}
            className={`${inputClasses} pl-11`}
            value={formData.email}
            onChange={(e) => setFormData({...formData, email: e.target.value})}
            onFocus={() => setFocused('email')}
            onBlur={() => setFocused('')}
          />
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, x: -20 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ delay: 0.3 }}
      >
        <div className="relative">
          <Phone className={`absolute left-3 top-3.5 w-5 h-5 transition-colors duration-300 ${focused === 'phone' ? 'text-green-600' : 'text-slate-400'}`} />
          <input
            type="tel"
            placeholder={contactContent.fields.phone}
            className={`${inputClasses} pl-11`}
            value={formData.phone}
            onChange={(e) => setFormData({...formData, phone: e.target.value})}
            onFocus={() => setFocused('phone')}
            onBlur={() => setFocused('')}
          />
        </div>
      </motion.div>

      <motion.div
        initial={{ opacity: 0, x: -20 }}
        animate={{ opacity: 1, x: 0 }}
        transition={{ delay: 0.4 }}
      >
        <div className="relative">
          <MessageSquare className={`absolute left-3 top-3.5 w-5 h-5 transition-colors duration-300 ${focused === 'message' ? 'text-pink-600' : 'text-slate-400'}`} />
          <textarea
            placeholder={contactContent.fields.message}
            className={`${inputClasses} pl-11 min-h-[150px] resize-none`}
            value={formData.message}
            onChange={(e) => setFormData({...formData, message: e.target.value})}
            onFocus={() => setFocused('message')}
            onBlur={() => setFocused('')}
          />
        </div>
      </motion.div>

      <motion.button
        type="submit"
        className="w-full py-4 px-6 rounded-xl bg-gradient-to-r from-blue-600 via-purple-600 to-pink-600 text-white font-semibold shadow-xl hover:shadow-2xl transition-all duration-300 flex items-center justify-center gap-2 group"
        whileHover={{ scale: 1.02, y: -2 }}
        whileTap={{ scale: 0.98 }}
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.5 }}
      >
        {contactContent.button}
        <Send className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
      </motion.button>
    </form>
  );
}

type OfficeLocationProps = {
  name: string;
  city: string;
  email: string;
  phone?: string;
  website?: string;
  gradient: string;
  icon: React.ElementType;
  index: number;
};

function OfficeLocation({ name, city, email, phone, website, gradient, icon: Icon, index }: OfficeLocationProps) {
  const { content } = useLanguage();
  const { contact: contactContent } = content;
  const [isHovered, setIsHovered] = useState(false);

  return (
    <motion.div
      initial={{ opacity: 0, y: 30 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true }}
      transition={{ duration: 0.6, delay: index * 0.15 }}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
      className="relative group"
    >
      <motion.div
        className={`absolute -inset-2 bg-gradient-to-r ${gradient} rounded-3xl blur-xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
        animate={isHovered ? { scale: [1, 1.05, 1] } : {}}
        transition={{ duration: 2, repeat: Infinity }}
      />
      <div className="relative bg-white/80 backdrop-blur-sm rounded-3xl p-8 shadow-xl border-2 border-slate-200/50 group-hover:border-white/50 transition-all duration-500 overflow-hidden">
        <motion.div
          className="absolute inset-0 opacity-5"
          style={{ backgroundImage: `radial-gradient(circle at 50% 50%, currentColor 1px, transparent 1px)`, backgroundSize: '20px 20px' }}
          animate={{ backgroundPosition: ['0px 0px', '20px 20px'] }}
          transition={{ duration: 20, repeat: Infinity, ease: "linear" }}
        />
        <div className="relative z-10 space-y-6">
          <div className="flex items-center gap-3">
            <motion.div
              className={`w-12 h-12 rounded-xl bg-gradient-to-br ${gradient} flex items-center justify-center shadow-lg`}
              whileHover={{ rotate: [0, -10, 10, -10, 0], scale: 1.1 }}
              transition={{ duration: 0.5 }}
            >
              <Icon className="w-6 h-6 text-white" />
            </motion.div>
            <div>
              <h3 className="text-2xl font-bold text-slate-900">{name}</h3>
              <p className="text-sm text-slate-500 font-medium">{city}</p>
            </div>
          </div>

          <div className="space-y-4">
            {phone && (
              <motion.div whileHover={{ x: 5 }} className="flex items-start gap-3 p-3 rounded-xl bg-gradient-to-r from-slate-50 to-blue-50/50 group-hover:from-blue-50 group-hover:to-purple-50/50 transition-all duration-300">
                <div className={`mt-0.5 w-8 h-8 rounded-lg bg-gradient-to-br ${gradient} flex items-center justify-center flex-shrink-0 shadow-md`}><Phone className="w-4 h-4 text-white" /></div>
                <div>
                  <div className="text-xs font-semibold text-slate-500 uppercase tracking-wide mb-1">{contactContent.labels.phone}</div>
                  <a href={`tel:${phone}`} className="text-slate-700 font-medium hover:text-transparent hover:bg-gradient-to-r hover:from-blue-600 hover:to-purple-600 hover:bg-clip-text transition-all">{phone}</a>
                </div>
              </motion.div>
            )}
            <motion.div whileHover={{ x: 5 }} className="flex items-start gap-3 p-3 rounded-xl bg-gradient-to-r from-slate-50 to-purple-50/50 group-hover:from-purple-50 group-hover:to-pink-50/50 transition-all duration-300">
              <div className={`mt-0.5 w-8 h-8 rounded-lg bg-gradient-to-br ${gradient} flex items-center justify-center flex-shrink-0 shadow-md`}><Mail className="w-4 h-4 text-white" /></div>
              <div>
                <div className="text-xs font-semibold text-slate-500 uppercase tracking-wide mb-1">{contactContent.labels.email}</div>
                <a href={`mailto:${email}`} className="text-slate-700 font-medium hover:text-transparent hover:bg-gradient-to-r hover:from-blue-600 hover:to-purple-600 hover:bg-clip-text transition-all">{email}</a>
              </div>
            </motion.div>
            {website && (
               <motion.div whileHover={{ x: 5 }} className="flex items-start gap-3 p-3 rounded-xl bg-gradient-to-r from-slate-50 to-green-50/50 group-hover:from-green-50 group-hover:to-teal-50/50 transition-all duration-300">
                <div className={`mt-0.5 w-8 h-8 rounded-lg bg-gradient-to-br ${gradient} flex items-center justify-center flex-shrink-0 shadow-md`}><Globe className="w-4 h-4 text-white" /></div>
                <div>
                  <div className="text-xs font-semibold text-slate-500 uppercase tracking-wide mb-1">{contactContent.labels.website}</div>
                  <a href={`http://${website}`} target="_blank" rel="noopener noreferrer" className="text-slate-700 font-medium hover:text-transparent hover:bg-gradient-to-r hover:from-blue-600 hover:to-purple-600 hover:bg-clip-text transition-all">{website}</a>
                </div>
              </motion.div>
            )}
          </div>
        </div>
      </div>
    </motion.div>
  );
}

export default function ContactSection() {
  const { content } = useLanguage();
  const { contact: contactContent } = content;

  const locations = [
    { ...contactContent.locations.ubc, icon: Building2, gradient: 'from-blue-600 to-cyan-600' },
    { ...contactContent.locations.noor, icon: Building2, gradient: 'from-green-600 to-teal-600' },
    { ...contactContent.locations.consortium, icon: Layers, gradient: 'from-purple-600 to-pink-600' },
  ];

  return (
    <section id="contact" className="scroll-mt-20 relative overflow-hidden bg-gradient-to-br from-slate-50 via-white to-purple-50/30 py-20 px-4">
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <motion.div
          className="absolute top-20 -left-20 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"
          animate={{ x: [0, 100, 0], y: [0, -50, 0], scale: [1, 1.2, 1] }}
          transition={{ duration: 20, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute bottom-20 -right-20 w-96 h-96 bg-purple-400/10 rounded-full blur-3xl"
          animate={{ x: [0, -100, 0], y: [0, 50, 0], scale: [1, 1.3, 1] }}
          transition={{ duration: 25, repeat: Infinity, ease: "easeInOut" }}
        />
      </div>

      <div className="relative z-10 max-w-7xl mx-auto">
        <motion.div 
          className="mx-auto max-w-2xl text-center mb-16"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
        >
          <motion.div
            className="inline-flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-blue-600/10 to-purple-600/10 rounded-full mb-6 backdrop-blur-sm border border-blue-200/50"
            initial={{ scale: 0 }}
            whileInView={{ scale: 1 }}
            viewport={{ once: true }}
            transition={{ type: "spring", duration: 0.8 }}
          >
            <Mail className="w-4 h-4 text-blue-600" />
            <span className="text-sm font-semibold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
              {contactContent.nav.contact}
            </span>
          </motion.div>

          <h2 className="text-4xl sm:text-5xl font-bold text-slate-900 mb-4 relative">
            <motion.span
              className="inline-block"
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.6, delay: 0.2 }}
            >
              {contactContent.title}
            </motion.span>
            <motion.div
              className="absolute -bottom-2 left-1/2 -translate-x-1/2 h-1 bg-gradient-to-r from-blue-600 via-purple-600 to-pink-600 rounded-full"
              initial={{ width: 0 }}
              whileInView={{ width: "150px" }}
              viewport={{ once: true }}
              transition={{ duration: 0.8, delay: 0.4 }}
            />
          </h2>

          <p className="mt-6 text-lg text-slate-600">
            {contactContent.description}
          </p>
        </motion.div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 lg:gap-16">
          <motion.div
            initial={{ opacity: 0, x: -30 }}
            whileInView={{ opacity: 1, x: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.7 }}
            className="relative group"
          >
            <div className="relative bg-white/80 backdrop-blur-sm rounded-3xl border-2 border-slate-200/50 group-hover:border-white/50 transition-all duration-500 overflow-hidden">
              <div className="relative bg-gradient-to-r from-blue-600 via-purple-600 to-pink-600 px-8 py-6">
                <motion.div
                  className="absolute inset-0 opacity-20"
                  style={{ backgroundImage: `radial-gradient(circle at 50% 50%, white 1px, transparent 1px)`, backgroundSize: '20px 20px' }}
                  animate={{ backgroundPosition: ['0px 0px', '20px 20px'] }}
                  transition={{ duration: 20, repeat: Infinity, ease: "linear" }}
                />
                <h3 className="relative text-2xl font-bold text-white flex items-center gap-2">
                  <MessageSquare className="w-6 h-6" />
                  {contactContent.form.title}
                </h3>
              </div>
              <div className="p-8">
                <ContactForm />
              </div>
            </div>
          </motion.div>

          <motion.div
            className="space-y-8"
            initial={{ opacity: 0, x: 30 }}
            whileInView={{ opacity: 1, x: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.7 }}
          >
            {locations.map((loc, index) => (
              <OfficeLocation
                key={loc.name}
                name={loc.name}
                city={loc.city}
                email={loc.email}
                phone={loc.phone}
                website={loc.website}
                icon={loc.icon}
                gradient={loc.gradient}
                index={index}
              />
            ))}
          </motion.div>
        </div>
      </div>
    </section>
  );
}

    
    